<?php
require_once "config.php";
require_once "db_connect.php";
header("Content-Type: application/json");

// --- Step 0: Check DB connection ---
if (!$conn) {
    die(json_encode(["status"=>"error", "message"=>"Database connection failed"]));
}

// --- Step 1: Get inputs ---
$device_id  = $_GET['device_id'] ?? '';
$ssid       = $_GET['ssid'] ?? '';
$latitude   = $_GET['latitude'] ?? '';
$longitude  = $_GET['longitude'] ?? '';

if (empty($device_id)) {
    echo json_encode(["status" => "error", "message" => "Missing device ID"]);
    exit;
}

// --- Step 2: Validate device ---
$deviceQuery = $conn->prepare("SELECT staff_token, ssid FROM staff_devices WHERE device_id = ?");
if (!$deviceQuery) {
    die(json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]));
}
$deviceQuery->bind_param("s", $device_id);
$deviceQuery->execute();
$deviceResult = $deviceQuery->get_result();

if ($deviceResult->num_rows === 0) {
    echo json_encode(["status" => "unauthorized", "message" => "Device not registered"]);
    exit;
}
$device = $deviceResult->fetch_assoc();
$staff_token = $device['staff_token'];

// --- Step 3: Get staff info ---
$staffQuery = $conn->prepare("SELECT fullname, department, email FROM staff WHERE staff_token = ?");
if (!$staffQuery) {
    die(json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]));
}
$staffQuery->bind_param("s", $staff_token);
$staffQuery->execute();
$staffResult = $staffQuery->get_result()->fetch_assoc();

// --- Step 4: Check today's attendance ---
$date = date("Y-m-d");
$checkToday = $conn->prepare("SELECT * FROM attendance WHERE staff_token = ? AND date = ?");
if (!$checkToday) {
    die(json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]));
}
$checkToday->bind_param("ss", $staff_token, $date);
$checkToday->execute();
$exists = $checkToday->get_result()->fetch_assoc();

// --- Step 5: Office hours ---
$office = $conn->query("SELECT * FROM officehours LIMIT 1")->fetch_assoc();
$currentTime = date("H:i:s");
$status = "checkin";

// Determine check-out
if ($exists) {
    $status = "checkout";
    $checkOutQuery = $conn->prepare("SELECT id FROM attendance WHERE staff_token = ? AND date = ? AND status='checkout'");
    if (!$checkOutQuery) {
        die(json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]));
    }
    $checkOutQuery->bind_param("ss", $staff_token, $date);
    $checkOutQuery->execute();
    if ($checkOutQuery->get_result()->num_rows > 0) {
        echo json_encode(["status" => "exists", "message" => "Already checked out today"]);
        exit;
    }
}

// --- Step 6: Wi-Fi or geofence validation ---
$allowedSSID = $device['ssid'];
$withinGeofence = false;

// Optional geofence fallback
$officeLat = -6.7924;  
$officeLng = 39.2083;
$radiusMeters = 100;

if (!empty($latitude) && !empty($longitude)) {
    $earthRadius = 6371000;
    $latFrom = deg2rad($officeLat);
    $lonFrom = deg2rad($officeLng);
    $latTo = deg2rad($latitude);
    $lonTo = deg2rad($longitude);

    $latDelta = $latTo - $latFrom;
    $lonDelta = $lonTo - $lonFrom;

    $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) +
        cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
    $distance = $angle * $earthRadius;

    if ($distance <= $radiusMeters) {
        $withinGeofence = true;
    }
}

// Deny if not on allowed SSID and not within geofence
if ($ssid !== $allowedSSID && !$withinGeofence) {
    echo json_encode(["status" => "denied", "message" => "Not within allowed area"]);
    exit;
}

// --- Step 7: Insert attendance ---
$attendance_token = substr(md5(uniqid()), 0, 10);
$day_of_week = date("l");
$week_number = date("W");
$month = date("F");

$stmt = $conn->prepare("INSERT INTO attendance (attendance_token, staff_token, user_token, latitude, longitude, status, time, day_of_week, week_number, month, date) VALUES (?,?,?,?,?,?,?,?,?,?,?)");
if (!$stmt) {
    die(json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]));
}

$user_token = $staff_token; 
$stmt->bind_param(
    "sssssssssss",
    $attendance_token,
    $staff_token,
    $user_token,
    $latitude,
    $longitude,
    $status,
    $currentTime,
    $day_of_week,
    $week_number,
    $month,
    $date
);
$stmt->execute();

echo json_encode([
    "status" => "success",
    "action" => $status,
    "message" => ucfirst($status) . " successful",
    "fullname" => $staffResult['fullname'],
    "department" => $staffResult['department'],
    "email" => $staffResult['email']
]);

$conn->close();
?>
