<?php
// scanned.php
header("Content-Type: text/html; charset=UTF-8");
require_once "db_connect.php";

// --- Optional: enable strict mysqli errors during development ---
// mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

function die_with_db_error($msg) {
    global $con;
    $err = $con->error ?? '';
    die($msg . ($err ? " (DB error: {$err})" : ""));
}

// Get inputs and validate
$staffToken = $_GET['staff_token'] ?? "";
$userToken  = $_GET['user_token']  ?? "";
$latRaw     = $_GET['lat']         ?? "";
$lngRaw     = $_GET['lng']         ?? "";

if ($staffToken === "" || $userToken === "" || $latRaw === "" || $lngRaw === "") {
    die("Missing required parameters. Expected: staff_token, user_token, lat, lng");
}

$lat = floatval($latRaw);
$lng = floatval($lngRaw);

// --- Fetch supervisor and site details (including radius) ---
$sql = "SELECT s.id AS staff_id, s.username, si.site_token, si.site_name, si.latitude, si.longitude, si.radius
        FROM staff s
        JOIN sites si ON s.site_token = si.site_token
        WHERE s.staff_token = ? LIMIT 1";

$stmt = $con->prepare($sql);
if (!$stmt) {
    die_with_db_error("Failed preparing staff/site query");
}
$stmt->bind_param("s", $staffToken);
$stmt->execute();
$result = $stmt->get_result();
if (!$row = $result->fetch_assoc()) {
    $stmt->close();
    die("Invalid staff_token or staff not found for this token.");
}
$stmt->close();

// normalize site values
$siteLat  = isset($row['latitude'])  ? floatval($row['latitude'])  : 0.0;
$siteLng  = isset($row['longitude']) ? floatval($row['longitude']) : 0.0;
$siteToken= $row['site_token'] ?? '';
$siteName = $row['site_name'] ?? 'Unknown site';
$allowedDistance = isset($row['radius']) && is_numeric($row['radius']) ? floatval($row['radius']) : 100.0;

// --- Fetch scanned user info (try staff table by staff_token first, otherwise users table) ---
$userFullname = "Unknown User";
$userDept = "N/A";
$foundUser = false;

// try staff table (common case where QR holds staff_token)
$q1 = $con->prepare("SELECT fullname, department FROM staff WHERE staff_token = ? LIMIT 1");
if ($q1) {
    $q1->bind_param("s", $userToken);
    $q1->execute();
    $r1 = $q1->get_result();
    if ($u = $r1->fetch_assoc()) {
        $userFullname = $u['fullname'] ?? $userFullname;
        $userDept = $u['department'] ?? $userDept;
        $foundUser = true;
    }
    $q1->close();
} else {
    // if prepare failed, don't die here — we'll try users table below
    // but log DB error for debugging
    error_log("scanned.php: prepare(staff) failed: " . ($con->error ?? ''));
}

// if not found in staff, try users table (if you have it)
if (!$foundUser) {
    $q2 = $con->prepare("SELECT fullname, department FROM users WHERE user_token = ? LIMIT 1");
    if ($q2) {
        $q2->bind_param("s", $userToken);
        $q2->execute();
        $r2 = $q2->get_result();
        if ($u2 = $r2->fetch_assoc()) {
            $userFullname = $u2['fullname'] ?? $userFullname;
            $userDept = $u2['department'] ?? $userDept;
            $foundUser = true;
        }
        $q2->close();
    } else {
        // if this prepare failed too, log it
        error_log("scanned.php: prepare(users) failed: " . ($con->error ?? ''));
    }
}

// --- Haversine distance function ---
function haversine_m($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371000; // meters
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat/2) * sin($dLat/2)
         + cos(deg2rad($lat1)) * cos(deg2rad($lat2))
         * sin($dLon/2) * sin($dLon/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    return $earthRadius * $c;
}

$distance = haversine_m($lat, $lng, $siteLat, $siteLng);
$withinRange = ($distance <= $allowedDistance);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Scanned Result</title>
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <style>
        :root { --accent: #1da1f2; --card-bg: #fff; --muted: #666; }
        body { font-family: Inter, "Segoe UI", Roboto, Arial, sans-serif; background:#f4f7fb; margin:0; padding:20px; color:#222; }
        .container { max-width:780px; margin:20px auto; }
        .card { background:var(--card-bg); border-radius:12px; box-shadow:0 6px 18px rgba(13,40,70,0.06); padding:18px; margin-bottom:18px; }
        h1,h2 { margin:0 0 8px 0; color:#233447; }
        p { margin:6px 0; color:var(--muted); }
        .row { display:flex; gap:16px; align-items:flex-start; flex-wrap:wrap; }
        .col { flex:1; min-width:200px; }
        .site-title { font-size:18px; font-weight:700; color:#0b2b42; }
        .coords { font-size:13px; color:#444; }
        .status { padding:8px 12px; border-radius:8px; font-weight:600; display:inline-block; }
        .ok { background:#e6fbf0; color:#0a8a2a; border:1px solid rgba(10,138,42,0.08); }
        .nok { background:#fff6f6; color:#b02a2a; border:1px solid rgba(176,42,42,0.08); }
        .user-card { background: linear-gradient(90deg,var(--accent), #0d8ddb); color:#fff; padding:14px; border-radius:10px; text-align:center; }
        .user-card h3 { margin:0; font-size:20px; }
        .user-card p { margin:6px 0 0 0; font-size:14px; opacity:0.95; }
        .btn { background:var(--accent); color:#fff; border:none; padding:12px 18px; border-radius:8px; cursor:pointer; font-weight:700; font-size:15px; display:inline-block; }
        .btn:disabled { background:#bcdff6; cursor:not-allowed; }
        small.debug { display:block; margin-top:8px; color:#999; font-size:12px; }
    </style>
</head>
<body>
<div class="container">

    <!-- SITE INFO -->
    <div class="card">
        <div class="row">
            <div class="col">
                <div class="site-title">🏗️ <?= htmlspecialchars($siteName) ?></div>
                <div class="coords">Site token: <?= htmlspecialchars($siteToken) ?></div>
                <div class="coords">Site coords: <?= htmlspecialchars($siteLat) ?> , <?= htmlspecialchars($siteLng) ?></div>
                <div class="coords">Allowed radius: <?= htmlspecialchars($allowedDistance) ?> m</div>
            </div>

            <div class="col" style="text-align:right;">
                <div style="font-weight:600; color:#444;">Scan info</div>
                <div class="coords">Staff (scanner): <?= htmlspecialchars($staffToken) ?></div>
                <div class="coords">Scanned token: <?= htmlspecialchars($userToken) ?></div>
            </div>
        </div>
    </div>

    <!-- LOCATION + DISTANCE -->
    <div class="card">
        <h2>📍 Your Location</h2>
        <div class="row">
            <div class="col">
                <p class="coords">Device coords: <strong><?= htmlspecialchars($lat) ?> , <?= htmlspecialchars($lng) ?></strong></p>
                <p class="coords">Distance to site: <strong><?= round($distance, 2) ?> m</strong></p>
                <?php if ($withinRange): ?>
                    <div class="status ok">✅ Within site range</div>
                <?php else: ?>
                    <div class="status nok">❌ Too far (<?= round($distance,2) ?> m)</div>
                <?php endif; ?>
                <small class="debug">Haversine used for distance calculation</small>
            </div>

            <!-- USER CARD -->
            <div class="col">
                <div class="user-card">
                    <h3><?= htmlspecialchars($userFullname) ?></h3>
                    <p>Department: <?= htmlspecialchars($userDept) ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- SUBMIT BUTTON -->
    <div class="card" style="text-align:center;">
        <?php if ($withinRange): ?>
            <form method="post" action="submit_attendance.php">
                <input type="hidden" name="staff_token" value="<?= htmlspecialchars($staffToken) ?>">
                <input type="hidden" name="user_token"  value="<?= htmlspecialchars($userToken) ?>">
                <input type="hidden" name="site_token"  value="<?= htmlspecialchars($siteToken) ?>">
                <input type="hidden" name="lat" value="<?= htmlspecialchars($lat) ?>">
                <input type="hidden" name="lng" value="<?= htmlspecialchars($lng) ?>">
                <button type="submit" class="btn">✔ Submit Attendance</button>
            </form>
        <?php else: ?>
            <button class="btn" disabled>Cannot submit – out of range</button>
        <?php endif; ?>
    </div>

</div>
<script>
// how often to check location (in milliseconds)
const RELOAD_INTERVAL = 15000; // 15 seconds

function reloadWithNewLocation() {
  if (navigator.geolocation) {
    navigator.geolocation.getCurrentPosition(
      (pos) => {
        const lat = pos.coords.latitude;
        const lng = pos.coords.longitude;

        // keep existing parameters
        const urlParams = new URLSearchParams(window.location.search);
        const staff = urlParams.get("staff_token");
        const user  = urlParams.get("user_token");

        // reload with updated coordinates
        window.location.href = `scanned.php?staff_token=${staff}&user_token=${user}&lat=${lat}&lng=${lng}`;
      },
      (err) => {
        console.error("Location error:", err);
      }
    );
  } else {
    console.error("Geolocation not supported by this browser.");
  }
}

// repeat location check every RELOAD_INTERVAL
setInterval(reloadWithNewLocation, RELOAD_INTERVAL);
</script>

</body>
</html>
