<?php
header("Content-Type: application/json; charset=utf-8");
require_once "db_connect.php";

$response = ["success" => false, "message" => ""];

// Collect inputs
$staff_token      = $_POST['staff_token'] ?? '';
$site_token       = $_POST['site_token'] ?? '';
$supervisor_token = $_POST['supervisor_token'] ?? '';
$latitude         = $_POST['latitude'] ?? '';
$longitude        = $_POST['longitude'] ?? '';
$status           = $_POST['status'] ?? 'checkin';

// Validate required fields
if ($staff_token === '' || $site_token === '' || $latitude === '' || $longitude === '') {
    $response['message'] = "Missing required fields.";
    echo json_encode($response);
    exit;
}

// Get staff and site info
$stmt = $con->prepare("
    SELECT s.id AS staff_id, s.fullname, st.site_name, st.latitude AS site_lat, st.longitude AS site_lon, st.radius
    FROM staff s
    JOIN sites st ON s.site_token = st.site_token
    WHERE s.staff_token = ? AND st.site_token = ?
    LIMIT 1
");
$stmt->bind_param("ss", $staff_token, $site_token);
$stmt->execute();
$result = $stmt->get_result();
$staff = $result->fetch_assoc();
$stmt->close();

if (!$staff) {
    $response['message'] = "Staff not found or site invalid.";
    echo json_encode($response);
    exit;
}

// Calculate distance (Haversine formula in meters)
function haversine($lat1, $lon1, $lat2, $lon2) {
    $earth_radius = 6371000; // meters
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    $a = sin($dLat/2) * sin($dLat/2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLon/2) * sin($dLon/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    return $earth_radius * $c;
}

$distance = haversine($latitude, $longitude, $staff['site_lat'], $staff['site_lon']);

if ($distance > $staff['radius']) {
    $response['message'] = "❌ Outside site radius. Distance: " . round($distance, 2) . "m";
    echo json_encode($response);
    exit;
}

// Save attendance
$stmt = $con->prepare("
    INSERT INTO attendance (staff_id, site_token, supervisor_token, status, latitude, longitude, distance, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
");
$stmt->bind_param("isssddd",
    $staff['staff_id'],
    $site_token,
    $supervisor_token,
    $status,
    $latitude,
    $longitude,
    $distance
);

if ($stmt->execute()) {
    $response['success'] = true;
    $response['message'] = "✅ Attendance saved. Distance: " . round($distance, 2) . "m";
} else {
    $response['message'] = "DB error: " . $stmt->error;
}
$stmt->close();

echo json_encode($response);
?>
